% This script generates Fig. 12.

% We plot the average best quotes in the last episode for different values
% of sigma, both in tau = 1 and in tau = 2 (unconditionally)

% Additionally, the figure plots the value of the best quote in each case 
% in the theoretical benchmark (Nash equilibrium)

%Pre-allocate a 9*2 matrix with 9 rows corresponding to 9 values of sigma
%and 2 columns with the averages of a_min_1, a_min_2

mat_av = zeros(9,2);

%Pre-allocate a 9*2 matrix with 9 rows corresponding to 9 values of sigma
%and 2 columns with 1.96 times standard deviations/sqrt(observations) of a_min_1, a_min_2

mat_sd = zeros(9,2);

%Fill the matrix by loading the data on last episodes for every sigma.

for sigma = 1:9
% Create the name of the baseline case dataset to be loaded
    dataname = sprintf('Data/last_episodes_%d.txt', sigma);
     % Load the data
    last_episodes = readmatrix(dataname);

    %Compute a_min_1.
    a_min_1 = min([last_episodes(:,1) last_episodes(:,3)],[],2);
    %Compute a_min_2.
    a_min_2 = min([last_episodes(:,2) last_episodes(:,4)],[],2);
    %Fill the matrix with the average of a_min_1, a_min_2.
    mat_av(sigma,1) = mean(a_min_1);
    mat_av(sigma,2) = mean(a_min_2);
    %Fill the matrix with 1.96* std. deviation of each price, divided by the
    %square root of the number of observations.
    [n1,~] = size(last_episodes);
    mat_sd(sigma,1) = 1.96*std(a_min_1)/sqrt(n1);
    mat_sd(sigma,2) = 1.96*std(a_min_2)/sqrt(n1);
end

%We create a matrix with the theoretical values for each sigma of a_min_1,
%a_min_2_NT and a_min_2_NT. The indexing is the same as in "mat".

%We first build mat_theory_temp with the prices conditionally on trade vs.
%no trade.

mat_theory_temp = zeros(9,3);

%Parameters
N = 2;
vH=4;
vL=0;
grid_middle = 8;
grid_size= 69;  
tick= 0.1;      

for sigma=1:9

%We start with a_min_1.
%We compute the theoretical expected profit associated with price a_1.
%We have Pi(a) = Pr(v=vH) Pr(vH + l > a) (a-vH) + Pr(v=vL) Pr(vL + l > a) (a-vL)
%We compute Pi(a) for every price in the grid. 

%NE is a matrix with 2*grid_size+1 lines (all prices on the grid), first
%column is the price, second column the associated profit, third column a
%variable equal to 1 if a is a NE price.

NE = zeros(2*grid_size+1,3);

%Compute the profit for each price.
for i=1:2*grid_size+1
    a = (grid_middle - grid_size*tick - tick)+(i*tick);
    NE(i,1) = a;
    profit = 0.5*(1-normcdf(a-vH,0,sigma))*(a-vH) + 0.5*(1-normcdf(a-vL,0,sigma))*(a-vL);
    NE(i,2) = profit;
end

%Check for each price whether it can be a NE, using Lemma 3 in the Online
%Appendix. We start with the second price, all prices below 2.0 can never be a NE
%price anyway.

%We set NE(:,3) to 1. We will replace with a zero each price for which we
%find a profitable deviation (not quoting anything or undercutting).
NE(:,3) = ones(2*grid_size+1,1);
%We know the first price necessarily generates losses.
NE(1,3) = 0;
for i=2:2*grid_size+1
%Check that the profit is positive.    
if NE(i,2) < 0
   NE(i,3) = 0;
end

%Check that having 1/N times the profit is better than the full profit at a lower price.    
for j=1:i-1
    if NE(j,2) > (1/N)*NE(i,2)
    NE(i,3) = 0;
    end
end

end

%Find highest NE price.
index = find(NE(:,3)==1,1,'last');
a1 = (grid_middle - grid_size*tick - tick)+(index*tick);
%Record the a_min_1:
mat_theory_temp(sigma,1) = a1;

%We now compute a_min_2_NT.
%The method is exactly the same as for a_min_1, except that the probability
%that v = vH is no longer 1/2 but the Bayesian probability given the
%observation of no trade in tau = 1.

%We start by computing the posterior probability mu that v = vH.
%Note that the period 1 ask is still a1, computed on line 118.
mu = 0.5*normcdf(a1-vH,0,sigma)/(0.5*normcdf(a1-vH,0,sigma)+0.5*normcdf(a1-vL,0,sigma));

%NE is a matrix with 2*grid_size+1 lines (all prices on the grid), first
%column is the price, second column the associated profit, third column a
%variable equal to 1 if a is a NE price.

NE = zeros(2*grid_size+1,3);

%Compute the profit for each price.
for i=1:2*grid_size+1
    a = (grid_middle - grid_size*tick - tick)+(i*tick);
    NE(i,1) = a;
    profit = mu*(1-normcdf(a-vH,0,sigma))*(a-vH) + (1-mu)*(1-normcdf(a-vL,0,sigma))*(a-vL);
    NE(i,2) = profit;
end

%Check for each price whether it can be a NE, using Lemma 3 in the Online
%Appendix. We start with the second price, all prices below 2.0 can never be a NE
%price anyway.

%We set NE(:,3) to 1. We will replace with a zero each price for which we
%find a profitable deviation (not quoting anything or undercutting).
NE(:,3) = ones(2*grid_size+1,1);
%We know the first price necessarily generates losses.
NE(1,3) = 0;
for i=2:2*grid_size+1
%Check that the profit is positive.    
if NE(i,2) < 0
   NE(i,3) = 0;
end

%Check that having 1/N times the profit is better than the full profit at a lower price.    
for j=1:i-1
    if NE(j,2) > (1/N)*NE(i,2)
    NE(i,3) = 0;
    end
end

end

%Find highest NE price.
index = find(NE(:,3)==1,1,'last');
a2_NT = (grid_middle - grid_size*tick - tick)+(index*tick);
%Record the a_min_2_NT:
mat_theory_temp(sigma,2) = a2_NT;


%We finally compute a_min_2_T.
%The method is exactly the same as for a_min_1, except that the probability
%that v = vH is no longer 1/2 but the Bayesian probability given the
%observation of a trade in tau = 1.

%We start by computing the posterior probability mu that v = vH.
%Note that the period 1 ask is still a, computed on line 118.
mu = 0.5*(1-normcdf(a1-vH,0,sigma))/(0.5*(1-normcdf(a1-vH,0,sigma))+0.5*(1-normcdf(a1-vL,0,sigma)));

%NE is a matrix with 2*grid_size+1 lines (all prices on the grid), first
%column is the price, second column the associated profit, third column a
%variable equal to 1 if a is a NE price.

NE = zeros(2*grid_size+1,3);

%Compute the profit for each price.
for i=1:2*grid_size+1
    a = (grid_middle - grid_size*tick - tick)+(i*tick);
    NE(i,1) = a;
    profit = mu*(1-normcdf(a-vH,0,sigma))*(a-vH) + (1-mu)*(1-normcdf(a-vL,0,sigma))*(a-vL);
    NE(i,2) = profit;
end

%Check for each price whether it can be a NE, using Lemma 3 in the Online
%Appendix. We start with the second price, all prices below 2.0 can never be a NE
%price anyway.

%We set NE(:,3) to 1. We will replace with a zero each price for which we
%find a profitable deviation (not quoting anything or undercutting).
NE(:,3) = ones(2*grid_size+1,1);
%We know the first price necessarily generates losses.
NE(1,3) = 0;
for i=2:2*grid_size+1
%Check that the profit is positive.    
if NE(i,2) < 0
   NE(i,3) = 0;
end

%Check that having 1/N times the profit is better than the full profit at a lower price.    
for j=1:i-1
    if NE(j,2) > (1/N)*NE(i,2)
    NE(i,3) = 0;
    end
end

end

%Find highest NE price.
index = find(NE(:,3)==1,1,'last');
a2_T = (grid_middle - grid_size*tick - tick)+(index*tick);
%Record the a_min_2_T:
mat_theory_temp(sigma,3) = a2_T;

end

%We now build mat_theory, a 9 x 2 matrix where each row is a value of
%sigma, column 1 is the theoretical price in tau = 1, and column 2 the
%average theoretical price in tau = 2.

mat_theory = zeros(9,2);

for sigma = 1:9
a1 = mat_theory_temp(sigma,1);
mat_theory(sigma,1) = a1;
p_trade = 0.5*(1-normcdf(a1-vH,0,sigma)) + 0.5*(1-normcdf(a1-vL,0,sigma));
mat_theory(sigma,2) = p_trade*mat_theory_temp(sigma,3) + (1-p_trade)*mat_theory_temp(sigma,2);
end


%% MEAN FINAL BEST QUOTES
figure();
hold on;

%Averages, a_min_1
plot(1:9, mat_av(:,1), 'Color',[0.0,0.0,1.0], 'LineWidth', 1);
%Confidence Interval
ci_a1 = fill([1:9, flip(1:9)], [(mat_av(:,1)-mat_sd(:,1))', flip((mat_av(:,1)+mat_sd(:,1))')], 'b');
set(ci_a1, 'facealpha', 0.3);  % make the confidence interval transparent
%Averages, a_min_2
plot(1:9, mat_av(:,2), 'Color',[1.0,0.0,0.0], 'LineWidth', 1);
%Confidence Interval
ci_a2 = fill([1:9, flip(1:9)], [(mat_av(:,2)-mat_sd(:,2))', flip((mat_av(:,2)+mat_sd(:,2))')], 'r');
set(ci_a2, 'facealpha', 0.3);  % make the confidence interval transparent

%Theoretical Benchmark
plot(1:9,mat_theory(:,1) ,'--o','Color',[0.0,0.0,1.0]) %a_min_1
plot(1:9,mat_theory(:,2) ,'--o','Color',[1.0,0.0,0.0]) %a_min_2

ylabel("Average Best Quote");
xlabel('$\sigma$', 'Interpreter', 'latex', 'FontSize', 14); 

legend('Simulations - $\bar{a}_1$', '95\% Confidence interval', 'Simulations - $\bar{a}_2$', '95\% Confidence interval','Nash Equilibrium - $\bar{a}_1$','Nash Equilibrium - $\bar{a}_2$','Location','Northwest','Interpreter','Latex','Fontsize',8);
legend('Box','off');
legend('NumColumns',1);
ylim([2,7]);
hold off;

%save png
filename = sprintf('/Figures/Fig_12.png');
saveas(gcf, [pwd filename]);
